/*
 * GStreamer
 *
 * unit test for flacparse
 *
 * Copyright (C) 2010 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/check/gstcheck.h>
#include "parser.h"

#define SRC_CAPS_TMPL  "audio/x-flac, framed=(boolean)false"
#define SINK_CAPS_TMPL  "audio/x-flac, framed=(boolean)true"

GstStaticPadTemplate sinktemplate = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (SINK_CAPS_TMPL)
    );

GstStaticPadTemplate srctemplate = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (SRC_CAPS_TMPL)
    );

/* some data */
static guint8 streaminfo_header[] = {
  0x7f, 0x46, 0x4c, 0x41, 0x43, 0x01, 0x00, 0x00,
  0x02, 0x66, 0x4c, 0x61, 0x43, 0x00, 0x00, 0x00,
  0x22, 0x12, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0a, 0xc4, 0x40, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00
};

static guint8 comment_header[] = {
  0x84, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00
};

static guint8 flac_frame[] = {
  0xff, 0xf8, 0xa9, 0x08, 0x00, 0x50, 0x18, 0x06,
  0x6a, 0x0c, 0xce, 0x13, 0x24, 0x19, 0x68, 0x00,
  0x46, 0x23, 0x08, 0xca, 0xcb, 0x58, 0x9c, 0x26,
  0x92, 0x30, 0xa6, 0x29, 0x8a, 0xca, 0xd1, 0x18,
  0xae, 0x26, 0x5c, 0x90, 0x60, 0xbf, 0x11, 0xad,
  0x43, 0x02, 0x06, 0x26, 0xbd, 0x35, 0xdd, 0xa3,
  0x11, 0xa6, 0x4d, 0x18, 0x8c, 0x9a, 0xe4, 0x62,
  0xd9, 0x23, 0x11, 0x8b, 0xcb, 0x56, 0x55, 0x45,
  0xc2, 0x18, 0x56, 0xa2, 0xe2, 0xe1, 0x18, 0x99,
  0x54, 0x98, 0x46, 0x4d, 0x08, 0x70, 0x9a, 0x64,
  0xc4, 0x18, 0x4f, 0x27, 0x64, 0x31, 0x66, 0x27,
  0x79, 0x19, 0x3c, 0x8c, 0x8c, 0xa3, 0x44, 0x18,
  0x23, 0xd2, 0x6b, 0x8b, 0x64, 0x8c, 0x21, 0x84,
  0xd6, 0x23, 0x13, 0x13, 0x2d, 0x44, 0xca, 0x5a,
  0x23, 0x09, 0x93, 0x25, 0x18, 0x10, 0x61, 0x38,
  0xb4, 0x60, 0x8f, 0x2c, 0x8d, 0x26, 0xb4, 0xc9,
  0xd9, 0x19, 0x19, 0x34, 0xd7, 0x31, 0x06, 0x10,
  0xc4, 0x30, 0x83, 0x17, 0xe2, 0x0c, 0x2c, 0xc4,
  0xc8, 0xc9, 0x3c, 0x5e, 0x93, 0x11, 0x8a, 0x62,
  0x64, 0x8c, 0x26, 0x23, 0x22, 0x30, 0x9a, 0x58,
  0x86, 0x04, 0x18, 0x4c, 0xab, 0x2b, 0x26, 0x5c,
  0x46, 0x88, 0xcb, 0xb1, 0x0d, 0x26, 0xbb, 0x5e,
  0x8c, 0xa7, 0x64, 0x31, 0x3d, 0x31, 0x06, 0x26,
  0x43, 0x17, 0xa3, 0x08, 0x61, 0x06, 0x17, 0xc4,
  0x62, 0xec, 0x4d, 0x4b, 0x2e, 0x2d, 0x4a, 0x94,
  0xa4, 0xc2, 0x31, 0x4c, 0x4c, 0x20, 0xc0, 0x83,
  0x14, 0x8c, 0x27, 0x8b, 0x31, 0x23, 0x2f, 0x23,
  0x11, 0x91, 0x94, 0x65, 0x1a, 0x20, 0xc2, 0x18,
  0x86, 0x51, 0x88, 0x62, 0x7c, 0x43, 0x2e, 0xa3,
  0x04, 0x18, 0x8c, 0x20, 0xc2, 0xf5, 0xaa, 0x94,
  0xc2, 0x31, 0x32, 0xd2, 0xb2, 0xa2, 0x30, 0xba,
  0x10, 0xc2, 0xb5, 0x89, 0xa5, 0x18, 0x10, 0x62,
  0x9a, 0x10, 0x61, 0x19, 0x72, 0x71, 0x1a, 0xb9,
  0x0c, 0x23, 0x46, 0x10, 0x62, 0x78, 0x81, 0x82,
  0x3d, 0x75, 0xea, 0x6b, 0x51, 0x8b, 0x61, 0x06,
  0x08, 0x62, 0x32, 0x5e, 0x84, 0x18, 0x27, 0x25,
  0xc2, 0x6a, 0x4b, 0x51, 0x31, 0x34, 0x5e, 0x29,
  0xa1, 0x3c, 0x4d, 0x26, 0x23, 0x10, 0xc2, 0x6b,
  0xb1, 0x0d, 0x11, 0xae, 0x46, 0x88, 0x31, 0x35,
  0xb1, 0x06, 0x08, 0x79, 0x7e, 0x4f, 0x53, 0x23,
  0x29, 0xa4, 0x30, 0x20, 0x30, 0x23, 0x5a, 0xb2,
  0xc8, 0x60, 0x9c, 0x93, 0x13, 0x17, 0x92, 0x98,
  0x46, 0x13, 0x54, 0x53, 0x08, 0xcb, 0x13, 0xa1,
  0x1a, 0x89, 0xe5, 0x46, 0x08, 0x18, 0x10, 0x30,
  0x9d, 0x68, 0xc2, 0x1c, 0x46, 0x46, 0xae, 0x62,
  0x1a, 0x46, 0x4e, 0x4d, 0x34, 0x8c, 0xbd, 0x26,
  0xc0, 0x40, 0x62, 0xc9, 0xa9, 0x31, 0x74, 0xa8,
  0x99, 0x52, 0xb0, 0x8c, 0xa9, 0x29, 0x84, 0x61,
  0x19, 0x54, 0x43, 0x02, 0x06, 0x04, 0x32, 0xe5,
  0x18, 0x21, 0x91, 0x8b, 0xf2, 0xcc, 0x10, 0x30,
  0x8e, 0x23, 0xc4, 0x76, 0x43, 0x08, 0x30, 0x83,
  0x08, 0x62, 0x6c, 0x4e, 0xe2, 0x35, 0x96, 0xd0,
  0x8e, 0x89, 0x97, 0x42, 0x18, 0x91, 0x84, 0x61,
  0x3c, 0x26, 0xa5, 0x2c, 0x4e, 0x17, 0x94, 0xb8,
  0xb5, 0xa4, 0xcb, 0x88, 0xc9, 0x84, 0x18, 0xb9,
  0x84, 0x19, 0x23, 0x2d, 0xa4, 0x64, 0x62, 0x18,
  0x86, 0x53, 0x93, 0xcb, 0x30, 0x8f, 0x2f, 0x93,
  0x55, 0xc4, 0xd7, 0x08, 0x62, 0xb8, 0x46, 0x84,
  0x68, 0xa3, 0x02, 0xaf, 0x33
};

static guint8 garbage_frame[] = {
  0xff, 0xff, 0xff, 0xff, 0xff
};


GST_START_TEST (test_parse_flac_normal)
{
  gst_parser_test_normal (flac_frame, sizeof (flac_frame));
}

GST_END_TEST;


GST_START_TEST (test_parse_flac_drain_single)
{
  gst_parser_test_drain_single (flac_frame, sizeof (flac_frame));
}

GST_END_TEST;


GST_START_TEST (test_parse_flac_drain_garbage)
{
  /* We always output the after frame garbage too because we
   * have no way of detecting it
   */
#if 0
  gst_parser_test_drain_garbage (flac_frame, sizeof (flac_frame),
      garbage_frame, sizeof (garbage_frame));
#endif
  guint8 frame[sizeof (flac_frame) + sizeof (garbage_frame)];

  memcpy (frame, flac_frame, sizeof (flac_frame));
  memcpy (frame + sizeof (flac_frame), garbage_frame, sizeof (garbage_frame));

  gst_parser_test_drain_single (frame, sizeof (frame));
}

GST_END_TEST;


GST_START_TEST (test_parse_flac_split)
{
  gst_parser_test_split (flac_frame, sizeof (flac_frame));
}

GST_END_TEST;


GST_START_TEST (test_parse_flac_skip_garbage)
{
  /* We always include the garbage into the frame because
   * we have no easy way for finding the real end of the
   * frame. The decoder will later skip the garbage
   */
#if 0
  gst_parser_test_skip_garbage (flac_frame, sizeof (flac_frame),
      garbage_frame, sizeof (garbage_frame));
#endif
  guint8 frame[sizeof (flac_frame) + sizeof (garbage_frame)];

  memcpy (frame, flac_frame, sizeof (flac_frame));
  memcpy (frame + sizeof (flac_frame), garbage_frame, sizeof (garbage_frame));

  gst_parser_test_normal (frame, sizeof (frame));
}

GST_END_TEST;


#define structure_get_int(s,f) \
    (g_value_get_int(gst_structure_get_value(s,f)))
#define fail_unless_structure_field_int_equals(s,field,num) \
    fail_unless_equals_int (structure_get_int(s,field), num)
/*
 * Test if the parser handles raw stream and codec_data info properly.
 */
GST_START_TEST (test_parse_flac_detect_stream)
{
  GstCaps *caps;
  GstStructure *s;
  const GValue *streamheader;
  GArray *bufarr;
  gint i;

  /* Push random data. It should get through since the parser should be
   * initialized because it got codec_data in the caps */
  caps = gst_parser_test_get_output_caps (flac_frame, sizeof (flac_frame),
      SRC_CAPS_TMPL);
  fail_unless (caps != NULL);

  /* Check that the negotiated caps are as expected */
  /* When codec_data is present, parser assumes that data is version 4 */
  GST_LOG ("flac output caps: %" GST_PTR_FORMAT, caps);
  s = gst_caps_get_structure (caps, 0);
  fail_unless (gst_structure_has_name (s, "audio/x-flac"));
  fail_unless_structure_field_int_equals (s, "channels", 1);
  fail_unless_structure_field_int_equals (s, "rate", 44100);
  fail_unless (gst_structure_has_field (s, "streamheader"));
  streamheader = gst_structure_get_value (s, "streamheader");
  fail_unless (G_VALUE_TYPE (streamheader) == GST_TYPE_ARRAY);
  bufarr = g_value_peek_pointer (streamheader);
  fail_unless (bufarr->len == 2);
  for (i = 0; i < bufarr->len; i++) {
    GstBuffer *buf;
    GValue *bufval = &g_array_index (bufarr, GValue, i);

    fail_unless (G_VALUE_TYPE (bufval) == GST_TYPE_BUFFER);
    buf = g_value_peek_pointer (bufval);
    if (i == 0) {
      fail_unless (GST_BUFFER_SIZE (buf) == sizeof (streaminfo_header));
      fail_unless (memcmp (buf, streaminfo_header, sizeof (streaminfo_header)));
    } else if (i == 1) {
      fail_unless (GST_BUFFER_SIZE (buf) == sizeof (comment_header));
      fail_unless (memcmp (buf, comment_header, sizeof (comment_header)));
    }
  }

  gst_caps_unref (caps);
}

GST_END_TEST;


static Suite *
flacparse_suite (void)
{
  Suite *s = suite_create ("flacparse");
  TCase *tc_chain = tcase_create ("general");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_parse_flac_normal);
  tcase_add_test (tc_chain, test_parse_flac_drain_single);
  tcase_add_test (tc_chain, test_parse_flac_drain_garbage);
  tcase_add_test (tc_chain, test_parse_flac_split);
  tcase_add_test (tc_chain, test_parse_flac_skip_garbage);

  /* Other tests */
  tcase_add_test (tc_chain, test_parse_flac_detect_stream);

  return s;
}


/*
 * TODO:
 *   - Both push- and pull-modes need to be tested
 *      * Pull-mode & EOS
 */

int
main (int argc, char **argv)
{
  int nf;

  Suite *s = flacparse_suite ();
  SRunner *sr = srunner_create (s);

  gst_check_init (&argc, &argv);

  /* init test context */
  ctx_factory = "flacparse";
  ctx_sink_template = &sinktemplate;
  ctx_src_template = &srctemplate;
  ctx_discard = 3;
  ctx_headers[0].data = streaminfo_header;
  ctx_headers[0].size = sizeof (streaminfo_header);
  ctx_headers[1].data = comment_header;
  ctx_headers[1].size = sizeof (comment_header);
  /* custom offsets, and ts always repeatedly 0 */
  ctx_no_metadata = TRUE;

  srunner_run_all (sr, CK_NORMAL);
  nf = srunner_ntests_failed (sr);
  srunner_free (sr);

  return nf;
}
